<?php

namespace Vuehoucine\Trustlicence\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Addon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use ZipArchive;

class AddonController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $addons = Addon::all();
        return view('trustlicence::addons.index', ['addons' => $addons]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (parse_url(url('/'))['host'] == 'localhost') {
            toastr()->error(__('Addons cannot be installed on local server'));
            return back();
        }
        if (!class_exists('ZipArchive')) {
            toastr()->error(__('ZipArchive extension is not enabled'));
            return back();
        }
        if (!$request->hasFile('addon_files')) {
            toastr()->error(__('Addon files required'));
            return back();
        }
        $addonZip = $request->addon_files;
        $addonZipFileExt = $addonZip->getClientOriginalExtension();
        if ($addonZipFileExt != "zip") {
            toastr()->error(__('File type not allowed'));
            return back();
        }
        try {
            $uploadPath = vFileUpload($addonZip, 'addons/temp/');
            $zip = new ZipArchive;
            $res = $zip->open($uploadPath);
            if ($res != true) {
                removeFile($uploadPath);
                toastr()->error(__('Could not open the zip file'));
                return back();
            }
            $dir = trim($zip->getNameIndex(0), '/');
            $addonsPath = base_path('addons/temp/');
            $thisAddonPath = base_path('addons/temp/' . $dir);
            if (File::exists($thisAddonPath)) {
                removeDirectory($thisAddonPath);
            }
            $res = $zip->extractTo($addonsPath);
            if ($res == true) {
                removeFile($uploadPath);
            }
            $zip->close();
            if (!File::exists($thisAddonPath . '/config.json')) {
                removeFile($thisAddonPath);
                toastr()->error(__('Config.json is missing'));
                return back();
            }
            $str = file_get_contents($thisAddonPath . '/config.json');
            $json = json_decode($str, true);
            if (strtolower(systemInfo()['name']) != $json['script_symbol']) {
                removeDirectory($thisAddonPath);
                toastr()->error(__('Invalid action request'));
                return back();
            }
            if (systemInfo()['version'] < $json['minimal_script_version']) {
                removeDirectory($thisAddonPath);
                toastr()->error(__('Addon require version ' . $json['minimal_script_version'] . ' or above'));
                return back();
            }
            $addonExist = Addon::where('symbol', $json['symbol'])->first();
            if ($addonExist) {
                toastr()->error(__('Addon is already exists'));
                return back();
            }
            if (!empty($json['remove_directories'])) {
                foreach ($json['remove_directories'] as $remove_directory) {
                    removeDirectory($remove_directory);
                }
            }
            if (!empty($json['remove_files'])) {
                foreach ($json['remove_files'] as $remove_file) {
                    removeFile($remove_file);
                }
            }
            if (!empty($json['directories'])) {
                foreach ($json['directories'][0]['assets'] as $assets_directory) {
                    makeDirectory($assets_directory);
                }
                foreach ($json['directories'][0]['files'] as $files_directory) {
                    makeDirectory(base_path($files_directory));
                }
            }
            if (!empty($json['assets'])) {
                foreach ($json['assets'] as $asset) {
                    File::copy(base_path($asset['root_directory']), $asset['update_directory']);
                }
            }
            if (!empty($json['files'])) {
                foreach ($json['files'] as $file) {
                    File::copy(base_path($file['root_directory']), base_path($file['update_directory']));
                }
            }
            if (!empty($json['sql_file'])) {
                if (file_exists(base_path($json['sql_file']))) {
                    DB::unprepared(file_get_contents(base_path($json['sql_file'])));
                }
            }
            $createAddon = Addon::create([
                "api_key" => 'api_key',
                "logo" => url($json['logo']),
                "name" => $json['name'],
                "symbol" => strtolower($json['symbol']),
                "version" => $json['version'],
                'action_text' => $json['action_text'],
                'action_link' => $json['action_link'],
                "status" => 1,
            ]);
            if ($createAddon) {
                removeDirectory($thisAddonPath);
                toastr()->success(__('Addon has been installed successfully'));
                return back();
            }
        } catch (\Exception$e) {
            toastr()->error($e->getMessage());
            return back();
        }

    }

    protected function addonPurchaseCodeValidate($purchaseCode, $symbol, $version)
    {
		$purchaseCode = $purchaseCode;
        $symbol = $symbol;
        $website = url('/');
        $version = $version;
        $client = new \GuzzleHttp\Client();
        $request = $client->get('https://trustlicence.com/api/v1/licence?purchaseCode=' . $purchaseCode . '&website=' .
            $website . '&symbol=' . $symbol . '&version=' . $version);
        $response = json_decode($request->getBody());
        return $response;
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Addon  $addon
     * @return \Illuminate\Http\Response
     */
    public function edit(Addon $addon)
    {
        return view('trustlicence::addons.edit', ['addon' => $addon]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Addon  $addon
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Addon $addon)
    {
        if (parse_url(url('/'))['host'] == 'localhost') {
            toastr()->error(__('Addons cannot be installed on local server'));
            return back();
        }
        if (!class_exists('ZipArchive')) {
            toastr()->error(__('ZipArchive extension is not enabled'));
            return back();
        }
        if (!$request->hasFile('addon_files')) {
            toastr()->error(__('Addon files required'));
            return back();
        }
        $addonZip = $request->addon_files;
        $addonZipFileExt = $addonZip->getClientOriginalExtension();
        if ($addonZipFileExt != "zip") {
            toastr()->error(__('File type not allowed'));
            return back();
        }
        try {
            $uploadPath = vFileUpload($addonZip, 'addons/temp/');
            $zip = new ZipArchive;
            $res = $zip->open($uploadPath);
            if ($res != true) {
                removeFile($uploadPath);
                toastr()->error(__('Could not open the zip file'));
                return back();
            }
            $dir = trim($zip->getNameIndex(0), '/');
            $addonsPath = base_path('addons/temp/');
            $thisAddonPath = base_path('addons/temp/' . $dir);
            if (File::exists($thisAddonPath)) {
                removeDirectory($thisAddonPath);
            }
            $res = $zip->extractTo($addonsPath);
            if ($res == true) {
                removeFile($uploadPath);
            }
            $zip->close();
            if (!File::exists($thisAddonPath . '/config.json')) {
                removeFile($thisAddonPath);
                toastr()->error(__('Config.json is missing'));
                return back();
            }
            $str = file_get_contents($thisAddonPath . '/config.json');
            $json = json_decode($str, true);
            if ($addon->symbol != $json['symbol']) {
                removeDirectory($thisAddonPath);
                toastr()->error(__('Invalid action request'));
                return back();
            }
            if (strtolower(systemInfo()['name']) != $json['script_symbol']) {
                removeDirectory($thisAddonPath);
                toastr()->error(__('Invalid action request'));
                return back();
            }
            if (systemInfo()['version'] < $json['minimal_script_version']) {
                removeDirectory($thisAddonPath);
                toastr()->error(__('Addon require version ' . $json['minimal_script_version'] . ' or above'));
                return back();
            }
            if (!empty($json['remove_directories'])) {
                foreach ($json['remove_directories'] as $remove_directory) {
                    removeDirectory($remove_directory);
                }
            }
            if (!empty($json['remove_files'])) {
                foreach ($json['remove_files'] as $remove_file) {
                    removeFile($remove_file);
                }
            }
            if (!empty($json['directories'])) {
                foreach ($json['directories'][0]['assets'] as $assets_directory) {
                    makeDirectory($assets_directory);
                }
                foreach ($json['directories'][0]['files'] as $files_directory) {
                    makeDirectory(base_path($files_directory));
                }
            }
            if (!empty($json['assets'])) {
                foreach ($json['assets'] as $asset) {
                    File::copy(base_path($asset['root_directory']), $asset['update_directory']);
                }
            }

            if (!empty($json['files'])) {
                foreach ($json['files'] as $file) {
                    File::copy(base_path($file['root_directory']), base_path($file['update_directory']));
                }
            }
            if (!empty($json['sql_file'])) {
                if (file_exists(base_path($json['sql_file']))) {
                    DB::unprepared(file_get_contents(base_path($json['sql_file'])));
                }
            }
            $updateAddon = $addon->update([
                "version" => $json['version'],
                'action_text' => $json['action_text'],
                'action_link' => $json['action_link'],
            ]);
            if ($updateAddon) {
                removeDirectory($thisAddonPath);
                toastr()->success(__('Addon has been updated successfully'));
                return back();
            }
        } catch (\Exception$e) {
            toastr()->error($e->getMessage());
            return back();
        }
    }

    protected function addonApiKeyValidate($api_key, $symbol)
    {
		$apikey = $api_key;
        $symbol = $symbol;
        $website = url('/');
        $client = new \GuzzleHttp\Client();
        $request = $client->get('https://trustlicence.com/api/v1/licence/details?apikey=' . $apikey . '&website=' .
            $website . '&symbol=' . $symbol);
        $response = json_decode($request->getBody());
        return $response;
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Addon  $addon
     * @return \Illuminate\Http\Response
     */
    public function destroy(Addon $addon)
    {
        return abort(404);
    }
}
